#include "Kernels.h"

#ifdef KERNEL_INFO_DISPLAY
int getKernelInfo(cl_kernel kernel){

  cl_int                 err;
  char*			 kernel_funct_name;
  cl_uint*		 kernel_num_args;
  cl_uint*		 kernel_ref_count;
  cl_context*            assoc_context;		// context associated with the kernel
  cl_program*		 assoc_program;		// program from which the kernel was created

  size_t sizeParam;

  // ----------------------------------------------------------------------------
  // Query Function name
  printf("\n-----------------------------------------------------------------------\n");
  err = clGetKernelInfo(kernel,CL_KERNEL_FUNCTION_NAME,0,NULL,&sizeParam);
  if (err != CL_SUCCESS){
	printf("Error: clGetKernelInfo() %d\n",err);
	fflush(stdout);
 	return EXIT_FAILURE;
  }

  kernel_funct_name = (char*) malloc(sizeof(char) * sizeParam);
  err = clGetKernelInfo(kernel,CL_KERNEL_FUNCTION_NAME,sizeParam,kernel_funct_name,NULL);
  if (err != CL_SUCCESS){
	printf("Error: clGetKernelInfo() %d\n",err);
	fflush(stdout);
	return EXIT_FAILURE;
  }

  printf("  %-45s: %s \n", "CL_KERNEL_FUNCTION_NAME", kernel_funct_name);
  free(kernel_funct_name);

  // ----------------------------------------------------------------------------
  // Query Number of arguments to kernel
  err = clGetKernelInfo(kernel,CL_KERNEL_NUM_ARGS,0,NULL,&sizeParam);
  if (err != CL_SUCCESS){
	printf("Error: clGetKernelInfo() %d\n",err);
	fflush(stdout);
	return EXIT_FAILURE;
  }

  kernel_num_args = (cl_uint*) malloc(sizeof(cl_uint) * sizeParam);
  err = clGetKernelInfo(kernel,CL_KERNEL_NUM_ARGS,sizeParam,kernel_num_args,NULL);
  if (err != CL_SUCCESS){
	printf("Error: clGetKernelInfo() %d\n",err);
	fflush(stdout);
	return EXIT_FAILURE;
  }

  printf("  %-45s: %u \n", "CL_KERNEL_NUM_ARGS", *kernel_num_args);
  free(kernel_num_args);

  // ----------------------------------------------------------------------------
  // Query Reference count of kernel
  /*
  https://www.khronos.org/registry/cl/specs/opencl-1.0.pdf
  CL_KERNEL_REFERENCE_COUNT
  The returned value should be considered stale (pasado, viejo).
  It is unsuitable for general use in applications.
  This feature is provided for identifying memory leaks.
  */
  err = clGetKernelInfo(kernel,CL_KERNEL_REFERENCE_COUNT,0,NULL,&sizeParam);
  if (err != CL_SUCCESS){
	printf("Error: clGetKernelInfo() %d\n",err);
	fflush(stdout);
 	return EXIT_FAILURE;
  }

  kernel_ref_count = (cl_uint*) malloc(sizeof(cl_uint) * sizeParam);
  err = clGetKernelInfo(kernel,CL_KERNEL_REFERENCE_COUNT,sizeParam,kernel_ref_count,NULL);
  if (err != CL_SUCCESS){
	printf("Error: clGetKernelInfo() %d\n",err);
	fflush(stdout);
	return EXIT_FAILURE;
  }

  printf("  %-45s: %u \n", "CL_KERNEL_REFERENCE_COUNT", *kernel_ref_count);
  free(kernel_ref_count);

  // ----------------------------------------------------------------------------
  // Query Context associated with kernel
  err = clGetKernelInfo(kernel,CL_KERNEL_CONTEXT,0,NULL,&sizeParam);
  if (err != CL_SUCCESS){
	printf("Error: clGetKernelInfo() %d\n",err);
	fflush(stdout);
 	return EXIT_FAILURE;
  }

  assoc_context = (cl_context*) malloc(sizeof(cl_context) * sizeParam);
  err = clGetKernelInfo(kernel,CL_KERNEL_CONTEXT,sizeParam,assoc_context,NULL);
  if (err != CL_SUCCESS){
	printf("Error: clGetKernelInfo() %d\n",err);
	fflush(stdout);
 	return EXIT_FAILURE;
  }

  printf("  %-45s: %x \n", "CL_KERNEL_CONTEXT", *assoc_context);
  free(assoc_context);

  // ----------------------------------------------------------------------------
  // Query Program from which the kernel was created
  err = clGetKernelInfo(kernel,CL_KERNEL_PROGRAM,0,NULL,&sizeParam);
  if (err != CL_SUCCESS){
	printf("Error: clGetKernelInfo() %d\n",err);
	fflush(stdout);
 	return EXIT_FAILURE;
  }

  assoc_program = (cl_program*) malloc(sizeof(cl_program) * sizeParam);
  err = clGetKernelInfo(kernel,CL_KERNEL_PROGRAM,sizeParam,assoc_program,NULL);
  if (err != CL_SUCCESS){
	printf("Error: clGetKernelInfo() %d\n",err);
	fflush(stdout);
 	return EXIT_FAILURE;
  }

  printf("  %-45s: %x \n", "CL_KERNEL_PROGRAM", *assoc_program);
  free(assoc_program);

  return CL_SUCCESS;
}
#endif

#ifdef KERNEL_WORK_GROUP_INFO_DISPLAY
int getKernelWorkGroupInfo(cl_kernel kernel, cl_device_id device){

  cl_int                 err;
  size_t*		 ker_wg_size;
  size_t*		 ker_pref_wg_size_multiple;
  cl_ulong* 		 ker_loc_mem_size;
  cl_ulong* 		 ker_pri_mem_size;
  size_t*		 ker_comp_wg_size;

  size_t sizeParam;

  // ----------------------------------------------------------------------------
  // Query Kernel work group size
  printf("\n-----------------------------------------------------------------------\n");
  err = clGetKernelWorkGroupInfo(kernel,device,CL_KERNEL_WORK_GROUP_SIZE,0,NULL,&sizeParam);
  if (err != CL_SUCCESS){
	printf("Error: clGetKernelWorkGroupInfo() %d\n",err);
	fflush(stdout);
 	return EXIT_FAILURE;
  }

  ker_wg_size = (size_t*) malloc(sizeof(size_t) * sizeParam);
  err = clGetKernelWorkGroupInfo(kernel,device,CL_KERNEL_WORK_GROUP_SIZE,sizeParam,ker_wg_size,NULL);
  if (err != CL_SUCCESS){
	printf("Error: clGetKernelWorkGroupInfo() %d\n",err);
	fflush(stdout);
 	return EXIT_FAILURE;
  }

  printf("  %-45s: %lu \n", "CL_KERNEL_WORK_GROUP_SIZE", ker_wg_size);
  free(ker_wg_size);

/*
  // CL_KERNEL_PREFERRED_WORK_GROUP_SIZE_MULTIPLE is supported from OpenCL 2.0
  // ----------------------------------------------------------------------------
  // Query Kernel preferred work group size multiple
  err = clGetKernelWorkGroupInfo(kernel,device,CL_KERNEL_PREFERRED_WORK_GROUP_SIZE_MULTIPLE,0,NULL,&sizeParam);
  if (err != CL_SUCCESS){printf("Error: clGetKernelWorkGroupInfo() %d\n",err); return EXIT_FAILURE;}

  ker_pref_wg_size_multiple = (size_t*) malloc(sizeof(size_t) * sizeParam);
  err = clGetKernelWorkGroupInfo(kernel,device,CL_KERNEL_PREFERRED_WORK_GROUP_SIZE_MULTIPLE,sizeParam,ker_pref_wg_size_multiple,NULL);
  if (err != CL_SUCCESS){printf("Error: clGetKernelWorkGroupInfo() %d\n",err); return EXIT_FAILURE;}

  printf("  %-45s: %lu \n", "CL_KERNEL_PREFERRED_WORK_GROUP_SIZE_MULTIPLE", ker_pref_wg_size_multiple);  
  free(ker_pref_wg_size_multiple);
*/

  // ----------------------------------------------------------------------------
  // Query Kernel local mem size
  err = clGetKernelWorkGroupInfo(kernel,device,CL_KERNEL_LOCAL_MEM_SIZE,0,NULL,&sizeParam);
  if (err != CL_SUCCESS){
	printf("Error: clGetKernelWorkGroupInfo() %d\n",err);
	fflush(stdout);
 	return EXIT_FAILURE;
  }

  ker_loc_mem_size = (size_t*) malloc(sizeof(size_t) * sizeParam);
  err = clGetKernelWorkGroupInfo(kernel,device,CL_KERNEL_LOCAL_MEM_SIZE,sizeParam,ker_loc_mem_size,NULL);
  if (err != CL_SUCCESS){
	printf("Error: clGetKernelWorkGroupInfo() %d\n",err);
	fflush(stdout);
 	return EXIT_FAILURE;
  }

  printf("  %-45s: %lu \n", "CL_KERNEL_LOCAL_MEM_SIZE (#bytes)", ker_loc_mem_size);  
  free(ker_loc_mem_size);

  // ----------------------------------------------------------------------------
  // Query Kernel private mem size
  err = clGetKernelWorkGroupInfo(kernel,device,CL_KERNEL_PRIVATE_MEM_SIZE,0,NULL,&sizeParam);
  if (err != CL_SUCCESS){
	printf("Error: clGetKernelWorkGroupInfo() %d\n",err);
	fflush(stdout);
 	return EXIT_FAILURE;
  }

  ker_pri_mem_size = (size_t*) malloc(sizeof(size_t) * sizeParam);
  err = clGetKernelWorkGroupInfo(kernel,device,CL_KERNEL_PRIVATE_MEM_SIZE,sizeParam,ker_pri_mem_size,NULL);
  if (err != CL_SUCCESS){
	printf("Error: clGetKernelWorkGroupInfo() %d\n",err);
	fflush(stdout);
 	return EXIT_FAILURE;
  }

  printf("  %-45s: %lu \n", "CL_KERNEL_PRIVATE_MEM_SIZE (#bytes)", ker_pri_mem_size);  
  free(ker_pri_mem_size);

  // ----------------------------------------------------------------------------
  // Query Kernel compiler work group size
  err = clGetKernelWorkGroupInfo(kernel,device,CL_KERNEL_COMPILE_WORK_GROUP_SIZE,0,NULL,&sizeParam);
  if (err != CL_SUCCESS){
	printf("Error: clGetKernelWorkGroupInfo() %d\n",err);
	fflush(stdout);
 	return EXIT_FAILURE;
  }

  ker_comp_wg_size = (size_t*) malloc(sizeof(size_t) * sizeParam);
  err = clGetKernelWorkGroupInfo(kernel,device,CL_KERNEL_COMPILE_WORK_GROUP_SIZE,sizeParam,ker_comp_wg_size,NULL);
  if (err != CL_SUCCESS){
	printf("Error: clGetKernelWorkGroupInfo() %d\n",err);
	fflush(stdout);
 	return EXIT_FAILURE;
  }

  printf("  %-45s: %lu %lu %lu\n", "CL_KERNEL_COMPILE_WORK_GROUP_SIZE", ker_comp_wg_size[0],  ker_comp_wg_size[1],  ker_comp_wg_size[2]);  
  free(ker_comp_wg_size);

  return CL_SUCCESS;
}
#endif

int setKernelArg(cl_kernel kernel, cl_uint num, size_t size, const void *ptr){
  cl_int err;
  err = clSetKernelArg(kernel,num,size,ptr);
  if (err != CL_SUCCESS){
	printf("Error: clSetKernelArg() %d\n", err);
	fflush(stdout);
	return EXIT_FAILURE;
  }
  return CL_SUCCESS;
}

int runKernel1D(cl_command_queue cmd_queue,
                cl_kernel        kernel,
		size_t		 gxDimSize,
		size_t		 lxDimSize,
                cl_ulong*        time_start,
		cl_ulong*	 time_stop){

  cl_int   err;
  cl_event event;
  //cl_ulong start;
  //cl_ulong stop;
  size_t gsize = gxDimSize;
  size_t lsize = lxDimSize;

  // Enqueue kernel
  err = clEnqueueNDRangeKernel(cmd_queue,kernel,1,NULL,&gsize,&lsize,0,NULL,&event);
  if (err != CL_SUCCESS){
	printf("Error: clEnqueueNDRangeKernel() %d\n", err);
	fflush(stdout);
	return EXIT_FAILURE;
  }

  // Ensure to have executed all enqueued tasks

  // clFinish commented out, as this causes slow down
  // Our command queue is in-order and this is not needed
  //clFinish(cmd_queue);

#ifdef CMD_QUEUE_PROFILING_ENABLE
  // Get start and stop time
  cl_ulong start;
  cl_ulong stop;
  clGetEventProfilingInfo(event,CL_PROFILING_COMMAND_START,sizeof(start),&start,NULL);
  clGetEventProfilingInfo(event,CL_PROFILING_COMMAND_END,  sizeof(stop),&stop, NULL);

  // Pass kernel exec time to calling function
  *time_start = start;
  *time_stop  = stop;
#endif

  clReleaseEvent(event);
  return CL_SUCCESS;
}


int runKernel2D(cl_command_queue cmd_queue,
                cl_kernel        kernel,
		size_t		 gxDimSize,
		size_t		 gyDimSize,
                size_t           lxDimSize,
  		size_t		 lyDimSize,
                cl_ulong*        time_start,
		cl_ulong*	 time_stop){

  cl_int   err;
  cl_event event;
  //cl_ulong start;
  //cl_ulong stop;
  size_t gsize[2] = {gxDimSize,gyDimSize};
  size_t lsize[2] = {lxDimSize,lyDimSize};

  // Enqueue kernel
  err = clEnqueueNDRangeKernel(cmd_queue,kernel,2,NULL,gsize,lsize,0,NULL,&event);
  if (err != CL_SUCCESS){
	printf("Error: clEnqueueNDRangeKernel() %d\n", err);
	fflush(stdout);
	return EXIT_FAILURE;
  }

  // Ensure to have executed all enqueued tasks

  // clFinish commented out, as this causes slow down
  // Our command queue is in-order and this is not needed
  //clFinish(cmd_queue);

#ifdef CMD_QUEUE_PROFILING_ENABLE
  // Get start and stop time
  cl_ulong start;
  cl_ulong stop;
  clGetEventProfilingInfo(event,CL_PROFILING_COMMAND_START,sizeof(start),&start,NULL);
  clGetEventProfilingInfo(event,CL_PROFILING_COMMAND_END,  sizeof(stop),&stop, NULL);

  // Pass kernel exec time to calling function
  *time_start = start;
  *time_stop  = stop;
#endif

  clReleaseEvent(event);
  return CL_SUCCESS;
}


int runKernel3D(cl_command_queue cmd_queue,
                cl_kernel        kernel,
		size_t		 gxDimSize,
		size_t		 gyDimSize,
                size_t 		 gzDimSize,
                size_t           lxDimSize,
  		size_t		 lyDimSize,
		size_t 		 lzDimSize,
                cl_ulong*        time_start,
		cl_ulong*	 time_stop){

  cl_int   err;
  cl_event event;
  //cl_ulong start;
  //cl_ulong stop;
  size_t gsize[3] = {gxDimSize,gyDimSize,gzDimSize};
  size_t lsize[3] = {lxDimSize,lyDimSize,lzDimSize};

  // Enqueue kernel
  err = clEnqueueNDRangeKernel(cmd_queue,kernel,3,NULL,gsize,lsize,0,NULL,&event);
  if (err != CL_SUCCESS){
	printf("Error: clEnqueueNDRangeKernel() %d\n", err);
	fflush(stdout);
	return EXIT_FAILURE;
  }

  // Ensure to have executed all enqueued tasks

  // clFinish commented out, as this causes slow down
  // Our command queue is in-order and this is not needed
  //clFinish(cmd_queue);

#ifdef CMD_QUEUE_PROFILING_ENABLE
  // Get start and stop time
  cl_ulong start;
  cl_ulong stop;
  clGetEventProfilingInfo(event,CL_PROFILING_COMMAND_START,sizeof(start),&start,NULL);
  clGetEventProfilingInfo(event,CL_PROFILING_COMMAND_END,  sizeof(stop),&stop, NULL);

  // Pass kernel exec time to calling function
  *time_start = start;
  *time_stop  = stop;
#endif

  clReleaseEvent(event);
  return CL_SUCCESS;
}
